package mcfall.raytracer;

import mcfall.raytracer.objects.RGBIntensity;

/**
 * <p>
 * A Material object describes the characteristics of a material
 * in terms of three reflection coefficients:
 * <ul>
 * 	<li>an <b>ambient</b> coefficient, describing how the material reflects
 *   ambient light
 *   </li>
 *   
 *   <li>a <b>specular</b> component, describing how the material reflects 
 *   "highlights
 *   </li>
 *   
 *   <li>
 *   a <b>diffuse</b> component
 *   </li>
 * </ul>
 * </p>
 * <p>
 * These components can be specified for each of the red, green
 * and blue components.
 * </p> 
 * @author mcfall
 *
 */
public class Material {

	private RGBIntensity ambient;
	private RGBIntensity diffuse;
	private RGBIntensity specular;
	
	/**
	 * Constructs a new material with the specified refelection
	 * coefficients
	 * @param aRed the proportion of red ambient light that is refelcted 
	 * @param aGreen the proportion of green ambient light that is reflected
	 * @param aBlue the proportion of blue ambient light that is reflected
	 * @param dRed the proportion of red diffuse light that is reflected
	 * @param dGreen the proportion of green diffuse light that is reflected
	 * @param dBlue the proportion of red diffuse light that is reflected
	 * @param sRed
	 * @param sGreen
	 * @param sBlue
	 */
	public Material (
			double aRed, double aGreen, double aBlue,
			double dRed, double dGreen, double dBlue,
			double sRed, double sGreen, double sBlue
	) {
		ambient = new RGBIntensity (aRed, aGreen, aBlue);
		diffuse = new RGBIntensity (dRed, dGreen, dBlue);
		specular = new RGBIntensity (sRed, sGreen, sBlue);
	}
	
	/**
	 * Constructs a new material with the specified refelection coefficients
	 * @param ambient an RGBIntensity triple describing the red, green and blue reflection of ambient light
	 * @param diffuse an RGBIntensity triple describing the red, green and blue reflection of diffuse light
	 * @param specular an RGBIntensity triple describing the red, green and blue reflection of specular light
	 */
	public Material(RGBIntensity ambient, RGBIntensity diffuse, RGBIntensity specular) {
		this (
			ambient.getRed(), ambient.getGreen(), ambient.getBlue(),
			diffuse.getRed(), diffuse.getGreen(), diffuse.getBlue(),
			specular.getRed(), specular.getGreen(), specular.getBlue()
		);
	}

	public RGBIntensity getAmbient () {
		return ambient;
	}

	public RGBIntensity getDiffuse() {
		return diffuse;
	}

	public RGBIntensity getSpecular() {
		return specular;
	}
}
